/*
 * Toolkit GUI, an application built for operating pinkRF's signal generators.
 *
 * Contact: https://www.pinkrf.com/contact/
 * Copyright © 2018-2024 pinkRF B.V
 * GNU General Public License version 3.
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License along with this program. If not, see https://www.gnu.org/licenses/
 *
 * Author: Iordan Svechtarov
 */

#ifndef RF_SYSTEM_1CHANNEL_H
#define RF_SYSTEM_1CHANNEL_H
#include "rf_system.h"
#include "rcm_class.h"
#include "gpioclassv2.h"
#include "miscellaneous.h"
#include <QThread>
#include <QSerialPortInfo>

#ifndef ALT_MODBUS_IMPLEMENTATION
#include "modbusserver.h"
#else
#include "modbusserver_v2.h"
#endif

#ifdef ETHERCAT
	#include "ethercat/module.h"
	#include "ethercat/signalsslots.h"
#endif

class RF_System_1channel : public QObject
{
	Q_OBJECT

public:
	RF_System_1channel();
	~RF_System_1channel();

	QList<QThread*> *threadList;

	RCM_Class *Remote_Command_Mode;
	ConfigHandler *config;
	ModbusServer *mbServer;

	/* EtherCAT stuff */
#ifdef ETHERCAT
	EtherCATSignalsSlots_1channel *ethercatSs;
	EtherCATModule ethercatModule;
#endif


private:
	QThread *channelThread;
	RF_Channel *channel;
	const int max_channel_count = 1;
	QList<bool> channels_working_state;
	QList<bool> PSU_readings_valid_state;

	RCM_MODE rcm_mode = RCM_MODE::RCM_MODE_OFF;

	bool channels_working = true;
	bool PSUs_readings_valid = true;

	QList<QSerialPort*> get_port_list();

#if defined(Q_OS_LINUX)

	GPIOClass *GPIO_control_external;
	GPIOClass *GPIO_control_SGx;
	GPIOClass *GPIO_LED_1;
	GPIOClass *GPIO_3v3_out;
	GPIOClass *GPIO_interlock_in;
	int last_GPIO_interlock_in_state = -1;

	QTimer *timer_actions;
	QTimer *timer_LED;
	LED_MODE led_mode = LED_MODE::LED_MODE_OFF;

	/* Non-slot GPIO functions */
	void poll_GPIO_interlock_in();
	void poll_GPIO_amplifier_mode();

	void set_GPIO_control_SGx(int value);
	void set_GPIO_control_external(int value);
	int get_GPIO_control_SGx();
	int get_GPIO_control_external();

	int determine_GPIO_amplifier_mode();

	void configure_LED_mode();

private slots:
	void LED_toggle();
	void timer_actions_slot();

	/* Private GPIO slots */
	void set_GPIO_amplifier_mode_enabled(int intrasys_num, bool enable);
	void handler_GPIO_PSU_timer();

#endif

private slots:
	void handler_remote_command_mode_get(RCM_MODE mode);

public slots:

	void handler_channel_working(int intrasys_num, bool enabled);
	void handler_PSU_readings_valid(int intrasys_num, bool valid);

	/* GPIO action slots */
#if defined(Q_OS_LINUX)
	void handler_LED_mode_set(int intrasys_num, LED_MODE mode);
#endif

signals:

	/* Raspberry Pi HAT signals */
#if defined(Q_OS_LINUX)
	void signal_GPIO_amplifier_mode_get(int intrasys_num, bool GPIO_amplifier_mode);
	void signal_GPIO_PSU_button_enable_get(int intrasys_num, bool GPIO_interlock_in);
#endif

	/* Modbus signals */
	void signal_set_generator_ready(bool state);
	void signal_SGx_communication_working(bool state);
	void signal_PSU_communication_working(bool state);

};

#endif //RF_SYSTEM_1CHANNEL_H
